unit QC_Main;
  {          QCard32.dll demonstrator.

    Shows how to use Stephen Murphy's QCard32.dll,
    a FreeWare, Windows-based, playing card library.
    Translated from VB to Delphi by Dale Cotton.
  }

  {
    Delphi Implementation Notes (Dale Cotton):

    1) This unit translates Stephen's VB DLL_Test into Delphi.
    For the most part this involved simple syntax substitutions;
    consequently this unit does not take advantage of the powerful
    OO techniques that Object Pascal provides over VB. The one
    exception I made was to define vars like BlockDragging and
    SourceCard, which were necessarily globals in VB as members
    of the TfrmMain class. This confines the scope of unexpected
    interactions to the TfrmMain methods...admittedly no big
    improvement.

    2) I have retained and extended Stephen's use of "Hungarian"
    prefix notation for variables, such as the n in nLoc which
    reminds us it is of type Integer. Admittedly Hungarian breaks
    down under OO when an object may be of type TSpeedButton or
    TPayrollDeduction. Hungarian is no less useful as a documentation
    tool for simple types, even if we use some other mechanism for
    object types.

    3) Prefixing QCard32. to QCard32 proc names is only absolutely
    necessary for the QCard32 EndDrag function, which has same name as
    a Win API call, however, using it in all cases adds safety and
    helps document code. When you go back to it six months from now
    you won't have to scratch your head as to where DrawCard comes
    from. Similarly,

    4) Prefixing Self. to all class member methods and data
    provides similar clarification. However, a better mechanism
    would distinguish between defined class fields and inherited
    ones. In the code below Self.X refers to a field defined in
    the ancestor class TForm, while Self.nOldX is a field defined
    directly in TfrmMain.

    5) Because VCL handles window client area painting more like
    Win API based programs than VB, it proved necessary to remove
    the drawing work handled in the VB demo's menu click event
    handlers from the actual menu click event handlers in this
    demo. See the note to the mnuDrawDrawCardClick procedure for
    details.

    6) My comments below will be marked (DC) to distinguish them
    from Stephen's, which are unmarked.
  }

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, Menus;

type
  TfrmMain = class(TForm)
    MainMenu1: TMainMenu;
    mnuDraw: TMenuItem;
    mnuDrawDrawCard: TMenuItem;
    mnuDrawDealCard: TMenuItem;
    mnuDrawDrawBack: TMenuItem;
    mnuDrawDrawSymbol: TMenuItem;
    mnuDrawRemoveCard: TMenuItem;
    N1: TMenuItem;
    mnuDrawExit: TMenuItem;
    mnuInfo: TMenuItem;
    mnuInfoCardInformation: TMenuItem;
    mnuDrag: TMenuItem;
    mnuDragDoDrag: TMenuItem;
    mnuHelp: TMenuItem;
    mnuHelpHowTo: TMenuItem;
    mnuHelpAbout: TMenuItem;
    Timer: TTimer;
    N2: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure mnuDrawDrawCardClick(Sender: TObject);
    procedure mnuDrawDealCardClick(Sender: TObject);
    procedure mnuDrawDrawBackClick(Sender: TObject);
    procedure mnuDrawDrawSymbolClick(Sender: TObject);
    procedure mnuDrawExitClick(Sender: TObject);
    procedure mnuInfoCardInformationClick(Sender: TObject);
    procedure mnuDrawRemoveCardClick(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
    procedure FormDblClick(Sender: TObject);
    procedure FormMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure FormMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure mnuDragDoDragClick(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure mnuHelpHowToClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure mnuHelpAboutClick(Sender: TObject);
  private
    bIsMenuCall: Boolean;     { see CardInformationClick event (DC) }
  { declare some test switches: }
    bDragDemo: Boolean;
    bSingleDragging: Boolean;
    bBlockDragging: Boolean;
    bMouseMoved: Boolean;
    nDrawSelection: Integer;

  { declare some card identifiers: }
    nSourceCard: Integer;
    nSourceArrayID: Integer;
    nSourceArrayPos: Integer;
    nDestCard: Integer;
    nDestArrayID: Integer;
    nOldX: Integer;
    nOldY: Integer;
    { nBlockMove called Temp and dynamically sized in VB code: }
    nBlockMove: array[0..51] of Integer;
    cBlockMove: Integer;  {nItems in VB code: BlockMove array count (DC) }
    nInformationCard: Integer;

  { to save mouse position for double click event: }
    xDblClick: Integer;
    yDblClick: Integer;

  { set up a two dimensional array four arrays to hold
    the numbers of the cards in each pile: }
    nCardArray: array[1..4, 1..52] of Integer;

  { set up a counter to go along with each pile: }
    nCounter: array[1..4] of Integer; { sizes of each pile (DC) }

    procedure DrawCard_Demo;
    procedure DealCard_Demo(bMenuCall: Boolean);
    procedure DrawBack_Demo;
    procedure DrawSymbol_Demo;
    procedure RemoveCard_Demo;
    procedure CardInformation_Demo(bMenuCall: Boolean);
  public
  end;

var
  frmMain: TfrmMain;

implementation

uses QCard32, QC_About;

{$R *.DFM}

procedure TfrmMain.DrawCard_Demo;
{ Draw all card images using DrawCard; this does not update
  any of the properties of the cards. (DC) }
var i, nLoc: Integer;
begin
  nLoc := (Self.ClientWidth - 4 * CARDWIDTH) div 5;

  for i := 1 to 13 do
    QCard32.DrawCard(Self.Handle, i, nLoc, 10 + ((i - 1) * OFFSET));

  for i := 14 to 26 do
    QCard32.DrawCard(Self.Handle, i, nLoc * 2 + CARDWIDTH, 10 + ((i - 14) * OFFSET));

  for i := 27 to 39 do
    QCard32.DrawCard(Self.Handle, i, nLoc * 3 + CARDWIDTH * 2, 10 + ((i - 27) * OFFSET));

  for i := 40 to 52 do
    QCard32.DrawCard(Self.Handle, i, nLoc * 4 + CARDWIDTH * 3, 10 + ((i - 40) * OFFSET));
end;

procedure TfrmMain.DealCard_Demo(bMenuCall: Boolean);
{ Deal cards in a diagonal line. See QCard32.hlp on differences
  between DealCard and DrawCard. (DC) }
var i, xOffset, yOffset: Integer;

{sub}procedure Shuffle;
  var i, j, k, nTemp: Integer;
  begin
    for i := 1 to 52 do
      nCardArray[1, i] := i;
    for i := 1 to 10 do
      for j := 1 to 52 do
      begin
        k := Random(52) + 1;
        nTemp := nCardArray[1, j];
        nCardArray[1, j] := nCardArray[1, k];
        nCardArray[1, k] := nTemp;
      end;
  end;

begin
  Shuffle;
  xOffset := (Self.ClientWidth - CARDWIDTH) div 51;
  yOffset := (Self.ClientHeight - CARDHEIGHT) div 51;
  if bMenuCall then { called from menu, so must Deal not Draw: (DC) }
    for i := 1 to 52 do
      QCard32.DealCard(Self.Handle, nCardArray[1, i], (i - 1) * xOffset, (i - 1) * yOffset)
  else  { can use faster method since DealCard already used: (DC) }
    for i := 1 to 52 do
      QCard32.DrawCard(Self.Handle, nCardArray[1, i], (i - 1) * xOffset, (i - 1) * yOffset);
end;

procedure TfrmMain.DrawBack_Demo;
{ Draw six piles of face down cards offsetting by 2 pixels
  up and over. }
var xLoc, i, j: Integer;
begin
  xLoc := (Self.ClientWidth - (6 * CARDWIDTH)) div 7;
  for i := 1 to 6 do
    for j := 1 to 4 do
      QCard32.DrawBack(Self.Handle, i,
        ((i - 1) * CARDWIDTH) + i * xLoc + ((j - 1) * 2),
          50 - ((j - 1) * 2));
end;

procedure TfrmMain.DrawSymbol_Demo;
{ Draw in one of each of the three pile symbols in the
  QCard image set. }
var xLoc, i: Integer;
begin
  xLoc := (Self.ClientWidth - (3 * CARDWIDTH)) div 4;
  for i := 1 to 3 do
    QCard32.DrawSymbol(Self.Handle, i, (i * xLoc) + ((i - 1) * CARDWIDTH), 50);
end;

procedure TfrmMain.RemoveCard_Demo;
{ Deal 13 cards then enable the timer to remove them. }
var i: Integer;
begin
  for i := 1 to 13 do
    QCard32.DealCard(Self.Handle, i, (Self.ClientWidth - CARDWIDTH) div 2, 10 + ((i - 1) * 16));

  Self.Timer.Enabled := True;
end;

procedure TfrmMain.CardInformation_Demo(bMenuCall: Boolean);
{ Demonstrates use of the QCard GetCardxxxx calls. (DC) }
const
   i: Integer = 0;
   xLoc: Integer = 0;
   yLoc: Integer = 0;
   nTextWdt: Integer = 0;

{sub-}procedure DoText(iCardNo: Integer);
  var
     sText: String;
     nTextHgt: Integer;
  begin
    nTextHgt := Trunc( Self.Canvas.TextHeight('X') * 1.25);
    sText := 'Card number is ' + IntToStr(iCardNo);
    Self.Canvas.TextOut(2, nTextHgt, sText);
    sText := 'Card color is ' + IntToStr(GetCardColor(iCardNo));
    Self.Canvas.TextOut(2, nTextHgt * 2, sText);
    sText := 'Card value (rank) is ' + IntToStr(GetCardValue(iCardNo));
    Self.Canvas.TextOut(2, nTextHgt * 3, sText);
    sText := 'Card suit is ' + IntToStr(GetCardSuit(iCardNo));
    Self.Canvas.TextOut(2, nTextHgt * 4, sText);
    sText := 'Card x location is ' + IntToStr(GetCardX(iCardNo));
    Self.Canvas.TextOut(2, nTextHgt * 5, sText);
    sText := 'Card y location is ' + IntToStr(GetCardY(iCardNo));
    Self.Canvas.TextOut(2, nTextHgt * 6, sText);
    if GetCardStatus(iCardNo) then
       sText := 'Card Status (face-up) is True'
    else
       sText := 'Card Status (face-up) value is False';
    Self.Canvas.TextOut(2, nTextHgt * 7, sText);
    sText := 'Main Form ClientWidth is ' + IntToStr(Self.ClientWidth);
    Self.Canvas.TextOut(2, nTextHgt * 8, sText);
    sText := 'Main Form ClientHeight is ' + IntToStr(Self.ClientHeight);
    Self.Canvas.TextOut(2, nTextHgt * 9, sText);
  end; { sub DoText }

begin
  if bMenuCall then { Called by menu click not OnPaint (DC) }
  begin
    nTextWdt := Trunc(Self.Canvas.TextWidth('Main Form ClientHeight is 9999'));
    { pick a random card: }
    i := Random(52) + 1;

    { pick a random location in the form --
      Int((upperbound - lowerbound + 1) * Rnd + lowerbound): }
    xLoc := Random((Self.ClientWidth - CARDWIDTH) - nTextWdt + 1) + nTextWdt;
    yLoc := Random((Self.ClientHeight - CARDHEIGHT) {- 150} + 1) {+ 150};
  end;

  { set current information card for Paint event: }
  Self.nInformationCard := i;

  { deal the card: }
  QCard32.DealCard(Self.Handle, i, xLoc, yLoc);

  { draw in the text information: }
  DoText(Self.nInformationCard);
end;

procedure TfrmMain.FormCreate(Sender: TObject);
begin
{ try to fire up the DLL. A False return value indicates problems: }
  if not QCard32.InitializeDeck(Self.Handle) then
  begin { The following error reports "Some other application is
          using QCard.DLL in 16-bit. This conflict does not apply
          in 32-bit. 32-bit allows multiple apps to use QCard.dll. (DC) }
    ShowMessage('Sorry. Unable to locate or open QCards32.DLL');
    Self.Close;
  end;

  Application.HelpFile :=
   LowerCase(ExtractFilePath(Application.ExeName))
    + Application.HelpFile;
  Randomize;

{ make form 3/4s of full screen: (DC) }
  Self.Width := (Screen.Width div 4) * 3;
  Self.Height := (Screen.Height div 4) * 3;

{ make some initial assigns:
  (actually unnecessary in Delphi which automatically
   initializes member vars to False/0) (DC) }
  Self.bDragDemo := False;
  Self.bSingleDragging := False;
  Self.bBlockDragging := False;
  Self.bMouseMoved := False;
  Self.nDrawSelection := 0;
  Self.nInformationCard := 0;
end;

procedure TfrmMain.mnuDrawDrawCardClick(Sender: TObject);
begin
  { User may have interrupted RemoveCard_Demo before
    it completed: (DC) }
  Self.Timer.Enabled := False;
  { Refresh first clears the main window client area,
    then sends a Paint message. The OnPaint event
    handler below uses nDrawSelection to know to
    call DrawCard_Demo to do the actual work. This
    round-about is necessary so that the program can
    respond to external events like a user over-lapping
    the main window then revealing it or the user dragging
    the resizing borders. Both of these invoke the
    OnPaint event, as well. (DC) }
  Self.nDrawSelection := 2;
  Self.Refresh;
end;

procedure TfrmMain.mnuDrawDealCardClick(Sender: TObject);
begin
  Self.Timer.Enabled := False;
  Self.nDrawSelection := 1;
  Self.bIsMenuCall := True;
  Self.Refresh;
  Self.bIsMenuCall := False;
end;

procedure TfrmMain.mnuDrawDrawBackClick(Sender: TObject);
begin
  Self.Timer.Enabled := False;
  Self.nDrawSelection := 3;
  Self.Refresh;
end;

procedure TfrmMain.mnuDrawDrawSymbolClick(Sender: TObject);
var xLoc, i: Integer;
begin
  Self.Timer.Enabled := False;
  Self.nDrawSelection := 4;
  Self.Refresh;
end;

procedure TfrmMain.mnuDrawExitClick(Sender: TObject);
begin
  Self.Timer.Enabled := False;
  Self.Close;
end;

procedure TfrmMain.mnuInfoCardInformationClick(Sender: TObject);
begin
  Self.Timer.Enabled := False;
  Self.nDrawSelection := 5;
  Self.bIsMenuCall := True;
  Self.Refresh;
  Self.bIsMenuCall := False;
end;

procedure TfrmMain.mnuDrawRemoveCardClick(Sender: TObject);
begin
  Self.nDrawSelection := 0;
  Self.Refresh;
  Self.RemoveCard_Demo;
end;

procedure TfrmMain.TimerTimer(Sender: TObject);
const n: Integer = 13;
begin
{ remove cards one at a time don't forget to take them off
  in reverse order or you will have a mess: }
  QCard32.RemoveCard(Self.Handle, n);
  Dec(n);
  if (n = 0) then
  begin
    n := 13;
    Self.Timer.Enabled := False;
  end;
end;

procedure TfrmMain.FormDblClick(Sender: TObject);
var nThisSourceCard, nThisDestCard, xNew, yNew: Integer;
begin
{  You can process double clicks in a similar
   way to the ButtonUp event.

   The current mouse position is saved for us in the
   OnMouseDown event as Self.xDblClick and Self.yDblClick

   Instead of using the class variables Self.nSourceCard
   and Self.nDestCard, we will use two local variables
   nThisSourceCard and nThisDestCard.

   We need to do this because Windows processes OnMouseDown
   and OnMouseUp messages before it actually gets to
   the OnDblClick event. This will keep our current
   selections from being corrupted by one of the
   other events.

   We can use the PointInFreeCard function to determine
   if the mouse is within any card that is not blocked: }

  nThisSourceCard := QCard32.PointInFreeCard(Self.xDblClick, Self.yDblClick);
  if (nThisSourceCard <> 0) then { double click is within free card }
  begin
    Self.nSourceArrayID := QCard32.GetUser4(nThisSourceCard);
    { pick a destination pile according to original pile
      (red to red suit, black to black suit): }
    Case Self.nSourceArrayID of
      1: Self.nDestArrayID := 4;
      2: Self.nDestArrayID := 3;
      3: Self.nDestArrayID := 2;
      4: Self.nDestArrayID := 1;
    end;

    Self.nSourceArrayPos := QCard32.GetUser3(nThisSourceCard);
    (*Self.nSourceArrayID := QCard32.GetUser4(nThisSourceCard); *)
    { if this is the last card in a row, and not the only card
      in the row then move it over to the other "same color row"
      and adjust arrays and blocks: }
    if (Self.nSourceArrayPos > 1)
    and (Self.nSourceArrayPos = Self.nCounter[Self.nSourceArrayID]) then
    begin
      nThisDestCard := Self.nCardArray[Self.nDestArrayID, Self.nCounter[Self.nDestArrayID]];
      xNew := GetCardX(nThisDestCard);
      yNew := GetCardY(nThisDestCard);
      RemoveCard(Self.Handle, nThisSourceCard);
      DealCard(Self.Handle, nThisSourceCard, xNew, yNew + OFFSET);
      Self.nCounter[Self.nSourceArrayID] := Self.nCounter[Self.nSourceArrayID] - 1;
      AdjustCardBlocked(Self.nCardArray[Self.nSourceArrayID, Self.nCounter[Self.nSourceArrayID]], False);
      AdjustCardBlocked(Self.nCardArray[Self.nDestArrayID, Self.nCounter[Self.nDestArrayID]], True);
      Self.nCounter[Self.nDestArrayID] := Self.nCounter[Self.nDestArrayID] + 1;
      Self.nCardArray[Self.nDestArrayID, Self.nCounter[Self.nDestArrayID]] := nThisSourceCard;
      QCard32.SetUser3(nThisSourceCard, Self.nCounter[Self.nDestArrayID]);
      QCard32.SetUser4(nThisSourceCard, Self.nDestArrayID);
    end;
  end;
end;

procedure TfrmMain.FormMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
{ InitDrag returns the number of the card that contains the mouse,
  as well as setting up the drag operation. }
begin
{ save mouse x and y position for double click event: }
  Self.xDblClick := X;
  Self.yDblClick := Y;

  if Self.bDragDemo then
  begin
    Self.nSourceCard := QCard32.InitDrag(Self.Handle, X, Y);
    if (Self.nSourceCard = 0) then { no card selected, so: }
      QCard32.AbortDrag
    else
    begin
      { save old position for later use if the drag is invalid: }
      Self.nOldX := QCard32.GetCardX(Self.nSourceCard);
      Self.nOldY := QCard32.GetCardY(Self.nSourceCard);
      { if card is not blocked, it is a single drag;
        if it is blocked, it means we're doing a block drag: }
      if QCard32.GetCardBlocked(Self.nSourceCard) then
        Self.bBlockDragging := True
      else
        Self.bSingleDragging := True;
    end;
  end;
end;

procedure TfrmMain.FormMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
var i: Integer;
begin
  if Self.bSingleDragging then
    { if just a single card, it's number was set with InitDrag call: }
    QCard32.DoDrag(Self.Handle, X, Y)
  else if Self.bBlockDragging then
  begin
    { determine which pile we are dealing with: }
    Self.nSourceArrayID := QCard32.GetUser4(Self.nSourceCard);
    { determine the position of the first card in drag: }
    Self.nSourceArrayPos := QCard32.GetUser3(Self.nSourceCard);
    { determine how many cards we are moving: }
    Self.cBlockMove := Self.nCounter[Self.nSourceArrayID] - Self.nSourceArrayPos + 1;
    (*{ create an array to hold the numbers of the cards to move
     and fill the array starting at 0: }
    ReDim Temp(nItems)*)

    for i := Self.nSourceArrayPos to Self.nCounter[Self.nSourceArrayID] do
        Self.nBlockMove[i - Self.nSourceArrayPos] := Self.nCardArray[Self.nSourceArrayID, i];
    { put a temporary block on the last card being dragged: }
    QCard32.AdjustCardBlocked(Self.nBlockMove[Self.cBlockMove - 1], True);
    { pass the BlockDrag procedure the actual array, referencing it's
      first element. This acts as a "pointer" to the rest of
      the elements in the array in memory: }
    QCard32.BlockDrag(Self.Handle, Self.nBlockMove[0], Self.cBlockMove, X, Y);
    { let the OnMouseUp event know that it is OK to
      reference the Self.nBlockMove[0] array for this instance: }
    Self.bMouseMoved := True;
  end;
end;

procedure TfrmMain.FormMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
  { most of the code here involves relocating cards
    to their new home arrays. }
var
  xDelta, yDelta, xSource, ySource, xNew, yNew,
  nUnused, nSourceColor, nDestColor, i: Integer;
begin
  if (Self.bSingleDragging = True) then
  begin
    { end the drag operation and find out whom we are dropping in on: }
    Self.nDestCard := QCard32.EndDrag(Self.Handle, X, Y);
    nSourceColor := GetCardColor(Self.nSourceCard);
    nDestColor := GetCardColor(Self.nDestCard);
    { which array did we come from: }
    Self.nSourceArrayID := QCard32.GetUser4(Self.nSourceCard);
    Self.nSourceArrayPos := QCard32.GetUser3(Self.nSourceCard);
    { which array are we joining?: }
    Self.nDestArrayID := QCard32.GetUser4(Self.nDestCard);

    { do some color testing, only allow drop if source and
      destination colors are the same and if nSourceCard is
      not the last card in it's pile. If nDestCard is 0,
      the Source Card was dropped at an invalid location: }
    if (Self.nDestCard = 0) or (nSourceColor <> nDestColor)
    or (Self.nSourceArrayPos = 1) then
    begin { if not a valid drop site, return drag: }
      ReturnDrag(Self.Handle, Self.nSourceCard, Self.nOldX, Self.nOldY);
      Self.bSingleDragging := False;
    end
    else
    begin { valid single drag/drop... proceed with relocation: }
      { reduce our old array counter: }
      Self.nCounter[Self.nSourceArrayID] := Self.nCounter[Self.nSourceArrayID] - 1;
      { add another to it's counter: }
      Self.nCounter[Self.nDestArrayID] := Self.nCounter[Self.nDestArrayID] + 1;
      { block our new neighbor: }
      AdjustCardBlocked(Self.nDestCard, True);
      { install our new array ID and position: }
      QCard32.SetUser3(Self.nSourceCard, Self.nCounter[Self.nDestArrayID]);
      QCard32.SetUser4(Self.nSourceCard, Self.nDestArrayID);
      { align with left side of card above us and down OFFSET (16): }
      xNew := GetCardX(Self.nDestCard);
      yNew := GetCardY(Self.nDestCard);
      RemoveCard(Self.Handle, Self.nSourceCard);
      DealCard(Self.Handle, Self.nSourceCard, xNew, yNew + OFFSET);
      { unblock last card in old array: }
      AdjustCardBlocked(Self.nCardArray[Self.nSourceArrayID, Self.nCounter[Self.nSourceArrayID]], False);
      { add ourselves to new array: }
      Self.nCardArray[Self.nDestArrayID, Self.nCounter[Self.nDestArrayID]] := Self.nSourceCard;
      Self.bSingleDragging := False;
    end
  end
  else if (Self.bBlockDragging = True) and (Self.bMouseMoved = True) then
  begin { we can employ the Self.nBlockMove[] array from OnMouseMove
  { as long as MouseMove actually occurred. }
    { end the drag and find out the destination card: }
    Self.nDestCard := EndBlockDrag(Self.Handle, Self.nBlockMove[0], Self.cBlockMove, X, Y);
    nSourceColor := GetCardColor(Self.nSourceCard);
    nDestColor := GetCardColor(Self.nDestCard);
    Self.nSourceArrayID := QCard32.GetUser4(Self.nSourceCard);
    Self.nSourceArrayPos := QCard32.GetUser3(Self.nSourceCard);
    Self.nDestArrayID := QCard32.GetUser4(Self.nDestCard);

    { do some color testing, only allow drop if source and
      destination colors are the same and if nSourceCard is
      not the last card in it's pile. If nDestCard is 0,
      the Source Card was dropped at an invalid location: }
    if (Self.nDestCard = 0) or (nSourceColor <> nDestColor)
    or (Self.nSourceArrayPos = 1) then { if not a valid drop site, return drag }
    begin
      { if not a valid drop site, return drag: }
      ReturnBlockDrag(Self.Handle, Self.nBlockMove[0], Self.cBlockMove, Self.nOldX, Self.nOldY);
      { remove temporary block on last card in block drag array: }
      AdjustCardBlocked(Self.nBlockMove[Self.cBlockMove - 1], False);
      Self.bBlockDragging := False;
      Self.bMouseMoved := False;
    end
    else
    begin
      { reduce our old array counter: }
      Self.nCounter[Self.nSourceArrayID] := Self.nCounter[Self.nSourceArrayID] - Self.cBlockMove;
      { block our new neighbor: }
      AdjustCardBlocked(Self.nDestCard, True);

      { this bit of code demonstrates how you can "fool" a drag operation
        to drag the item to a specific location. Usually, you pass the
        BlockDrag sub the x,y location of the mouse. If you first determine
        your current mouse position in relation to the object you are dragging
        you can add that difference (xDelta, yDelta) to the position you
        want to drag to, and pass those points to BlockDrag. We want to align
        with the left side of DestCard and down OFFSET (16) pixels from its top: }
      xNew := GetCardX(Self.nDestCard);
      yNew := GetCardY(Self.nDestCard);
      xSource := GetCardX(Self.nSourceCard);
      ySource := GetCardY(Self.nSourceCard);
      xDelta := X - xSource;
      yDelta := Y - ySource;
      nUnused := InitDrag(Self.Handle, X, Y);
      BlockDrag(Self.Handle, Self.nBlockMove[0], Self.cBlockMove,
       xNew + xDelta, yNew + OFFSET + yDelta);
      nUnused := EndBlockDrag(Self.Handle, Self.nBlockMove[0], Self.cBlockMove,
       xNew + xDelta, yNew + OFFSET + yDelta);

      for i := 0 to Self.cBlockMove - 1 do
      begin { install our new array IDs and positions: }
        Self.nCounter[Self.nDestArrayID] := Self.nCounter[Self.nDestArrayID] + 1;
        Self.nCardArray[Self.nDestArrayID, Self.nCounter[Self.nDestArrayID]] := Self.nBlockMove[i];
        QCard32.SetUser3(Self.nBlockMove[i], Self.nCounter[Self.nDestArrayID]);
        QCard32.SetUser4(Self.nBlockMove[i], Self.nDestArrayID);
      end;

      { unblock last card in old array: }
      AdjustCardBlocked(Self.nCardArray[Self.nSourceArrayID, Self.nCounter[Self.nSourceArrayID]], False);

      { remove temporary block on last card in block drag array: }
      AdjustCardBlocked(Self.nBlockMove[Self.cBlockMove - 1], False);

      Self.bBlockDragging := False;
      Self.bMouseMoved := False;
    end
  end
  else if (Self.bBlockDragging = True) and (Self.bMouseMoved = False) then
  begin { there was a MouseDown event but no MouseMove event }
    AbortDrag;
    Self.bBlockDragging := False;
  end;
end;

procedure TfrmMain.mnuDragDoDragClick(Sender: TObject);
var cxSpacer, i: Integer;
begin
  Self.nDrawSelection := 6;
  QCard32.SetDefaultValues;  { clear out any old card properties }
  Self.Refresh;

  cxSpacer := (Self.ClientWidth - 4 * CARDWIDTH) div 5;

  for i := 1 to 4 do { draw in pile marker symbols: }
      QCard32.DrawSymbol(Self.Handle, 1, cxSpacer * i + ((i - 1) * CARDWIDTH), 10);

  { Each pile has it's own array identifying the cards;
    each pile has a counter to maintain the pile;
    each card uses it's User3 and User4 properties to
    store which array it belongs to and what position
    it's in within the array. This makes dragging and
    dropping easier. }

 { deal first pile and set up array: }
  for i := 1 to 13 do
  begin
    QCard32.DealCard(Self.Handle, i, cxSpacer, 10 + ((i - 1) * OFFSET));
    Self.nCardArray[1, i] := i;
    QCard32.SetUser3(i, i);   { card's position in array }
    QCard32.SetUser4(i, 1);   { array ID }
    if (i < 13) then { block all cards except the one on top }
      QCard32.AdjustCardBlocked(i, True);
  end;

  Self.nCounter[1] := 13;   { there are 13 cards per pile }

  for i := 14 to 26 do
  begin
    QCard32.DealCard(Self.Handle, i, (cxSpacer * 2) + CARDWIDTH, 10 + ((i - 14) * OFFSET));
    Self.nCardArray[2, i - 13] := i;
    QCard32.SetUser3(i, i - 13);  { card's position in array }
    QCard32.SetUser4(i, 2);       { array ID }
    if (i < 26) then
      QCard32.AdjustCardBlocked(i, True);
  end;
  Self.nCounter[2] := 13;

  for i := 27 to 39 do
  begin
    QCard32.DealCard(Self.Handle, i, (cxSpacer * 3) + (2 * CARDWIDTH), 10 + ((i - 27) * OFFSET));
    Self.nCardArray[3, i - 26] := i;
    QCard32.SetUser3(i, i - 26);  { card's position in array }
    QCard32.SetUser4(i, 3);       { array ID }
    if (i < 39) then
      QCard32.AdjustCardBlocked(i, True);
  end;
  Self.nCounter[3] := 13;

  for i := 40 to 52 do
  begin
    QCard32.DealCard(Self.Handle, i, (cxSpacer * 4) + (3 * CARDWIDTH), 10 + ((i - 40) * OFFSET));
    Self.nCardArray[4, i - 39] := i;
    QCard32.SetUser3(i, i - 39);
    QCard32.SetUser4(i, 4);
    if (i < 52) then
      QCard32.AdjustCardBlocked(i, True);
  end;
  Self.nCounter[4] := 13;
  Self.bDragDemo := True;
end;

procedure TfrmMain.FormPaint(Sender: TObject);
var x, y, i, j: Integer;
begin
  { Even when the AutoRedraw property for your form is set
    to TRUE, Windows will not redraw any of your cards for
    you. You must handle the redrawing in the Paint Event.
    In a normal card game, your Paint Event will look a lot
    like Case 6 below. }

  Case nDrawSelection of
    1: Self.DealCard_Demo(Self.bIsMenuCall);
    2: Self.DrawCard_Demo;
    3: Self.DrawBack_Demo;
    4: Self.DrawSymbol_Demo;
    5: Self.CardInformation_Demo(Self.bIsMenuCall);
    6: begin
        for i := 1 to 4 do
          for j := 1 to Self.nCounter[i] do
          begin
            { if resizable window then you should probably
              check if window has been resized, and if so
              recalc x and y to new client area then use
              DealCard not DrawCard: (DC) }
            x := QCard32.GetCardX(Self.nCardArray[i, j]);
            y := QCard32.GetCardY(Self.nCardArray[i, j]);
            QCard32.DrawCard(Self.Handle, Self.nCardArray[i, j], x, y);
          end;
       end;
  end;
end;

procedure TfrmMain.FormResize(Sender: TObject);
begin
  Self.Refresh; { clears previous client area content
                  and initiates a FormPaint event. (DC) }
end;

procedure TfrmMain.mnuHelpHowToClick(Sender: TObject);
begin
  Application.HelpCommand(HELP_CONTENTS, 0);
end;

procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  Application.HelpCommand(HELP_QUIT, 0);
end;

procedure TfrmMain.mnuHelpAboutClick(Sender: TObject);
begin
  frmAbout.ShowModal;
end;

end.
